#: Makefile/GraphViz.pm
#: Plot the detailed structure of Makefiles
#:   using GraphViz
#: v0.01
#: Copyright (c) 2005 Agent Zhang
#: 2005-09-30 2005-10-01

package Makefile::GraphViz;

use strict;
use warnings;

use GraphViz;
use base 'Makefile::Parser';

our $VERSION = '0.01';

my $node_attr =
{
    shape => 'box',
    style => 'filled',
    fillcolor => 'yellow',
};

my $edge_attr =
{
    color => 'red',
};

my %args = (
    layout => 'dot',
    ratio => 'auto',
    node => $node_attr,
    edge => $edge_attr,
);

our %Nodes;

sub plot {
    my ($self, $root_name, $gv) = @_;
    my $root = ($root_name and ref $root_name) ? $root_name : ($self->target($root_name));

    if (!$gv) {
        $gv = GraphViz->new(%args);
        %Nodes = ();
    }
    #warn $gv;
    if (!$Nodes{$root_name}) {
        (my $short_name = $root_name) =~ s/.*(.{5}[\\\/].*)$/...$1/o;
        $short_name =~ s/\\/\//g;
        #warn $short_name, "\n";
        $gv->add_node($root_name, label => $short_name);
        $Nodes{$root_name} = 1;
    } else {
        return $gv;
    }
    #warn "GraphViz: $gv\n";
    return $gv if !$root;

    my @depends = $root->depends;
    foreach (@depends) {
        #warn "$_\n";
        $gv->add_edge($_ => $root_name);
        $self->plot($_, $gv);
    }
    #warn "END\n";
    #warn "GraphViz: $gv\n";
    return $gv;
}

1;
__END__

=head1 NAME

Makefile::GraphViz - Plot the Detailed Structure of Makefiles Using GraphViz                        

=head1 SYNOPSIS

  use Makefile::GraphViz;

  $parser = Makefile::GraphViz->new;
  $parser->parse('Makefile');

  # plot the tree rooted at the install target in Makefile:
  $gv = $parser->plot('install');  # A GraphViz object returned.
  $gv->as_png('install.png');

  # plot the tree rooted at the default target in Makefile:
  $gv = $parser->plot;
  $gv->as_png('default.png');

  # plot the forest consists of all the targets in Makefile:
  $gv = $parser->plot_all;
  $gv->as_png('default.png');

  # you can also invoke all the methods inherited from the Makefile::Parser class:
  @targets = $parser->targets;

=head1 DESCRIPTION

This module uses L<Makefile::Parser> to render user's Makefiles via the amazing
L<GraphViz> module. Before I decided to write this thing, there had been already a
CPAN module named L<GraphViz::Makefile> which did the same thing. However, the
pictures generated by L<GraphViz::Makefile> is oversimplified in my opinion, So
a much complex one is still needed.

B<IMPORTANT!>
This stuff is highly experimental and is currently at B<ALPHA> stage, so
production use is strongly discouraged. Anyway, I have the plan to 
improve this stuff unfailingly.

=head1 The Makefile::GraphViz Class

This class is a subclass inherited from L<Makefile::Parser>. So all the methods (and
hence all the functionalities) provided by L<Makefile::Parser> are accessable here.
Additionally this class also provides some more methods on its own right.

=head1 METHODS

=over

=item plot($target_name)

This method is essential to the class. Users invoke this method to plot the specified
Makefile target. If the argument is absent, the default target in the Makefile will
be used. It will return a L<GraphViz> object, on which you can later call the
-E<gt>as_png or -E<gt>as_text method to obtain the final graphical output.

The argument can both be the target's name and a Makefile::Target object. If the
given target can't be found in Makefile, the target will be plotted separately.

=back

=head2 EXPORT

None by default.

=head1 CODE COVERAGE

I use L<Devel::Cover> to test the code coverage of my tests, below is the 
L<Devel::Cover> report on this module test suite.

    ---------------------------- ------ ------ ------ ------ ------ ------ ------
    File                           stmt   bran   cond    sub    pod   time  total
    ---------------------------- ------ ------ ------ ------ ------ ------ ------
    .../lib/Makefile/GraphViz.pm  100.0   87.5   33.3  100.0  100.0  100.0   93.5
    Total                         100.0   87.5   33.3  100.0  100.0  100.0   93.5
    ---------------------------- ------ ------ ------ ------ ------ ------ ------

=head1 SEE ALSO

L<Makefile::Parser>, L<GraphViz::Makefile>.

=head1 AUTHOR

Agent Zhang, E<lt>agent2002@126.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2005 Agent Zhang.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
