/* Created on 06.09.2006 */

package de.memowe.sbc;

import de.memowe.sbc.kernel.*;
import java.util.Vector;
import java.util.regex.*;

/**
 * Simple Blog Code Parser.
 * 
 * Simple Blog Code is a simple markup language. You can use it for guest books,
 * blogs, wikis, boards and various other web applications. It produces valid
 * and semantic (X)HTML from input and is patterned on that tiny usenet markups
 * like *bold* and _underline_.
 * 
 * HTML::SBC tries to give useful error messages and guess the right translation
 * even with invalid input. It will always produce valid (X)HTML.
 * 
 * @author Mirko Westermeier <mail@memowe.de>
 * @version 0.1 (based on HTML::SBC 0.14)
 */
public class SBCParser {

    /* DATA SECTION */
    
    // static SBC parse interface
    private static SBCParser staticParser;

    // parser state...
    private String  text;
    private String  result;
    private String  attribute;
    private Vector  errors;
    private Istack  istack;
    private int     qstack;
    private int     line;

    // private istack struct
    private class Istack {
        public int emphasis, strong, hyperlink;
        public Istack() { emphasis = strong = hyperlink = 0; }
    }

    // parser object state...
    private int             language;
    private boolean         imageSupport;
    private ErrorReceiver   errorCallback;
    private URLChecker      linkcheckCallback;
    private URLChecker      imgcheckCallback;

    /* CONSTRUCTOR SECTION */

    /**
     * Constructs a SBC parser object.
     * 
     * @param language language for error messages
     * @param imageSupport whether image markup is translated
     * @param errorCallback callback object for error messages
     * @param linkcheckCallback callback object for link url validation
     * @param imgcheckCallback callback object for image url validation
     */
    public SBCParser(int language, boolean imageSupport,
            ErrorReceiver errorCallback, URLChecker linkcheckCallback,
            URLChecker imgcheckCallback) {
        this.language           = language;
        this.imageSupport       = imageSupport;
        this.errorCallback      = errorCallback;
        this.linkcheckCallback  = linkcheckCallback;
        this.imgcheckCallback   = imgcheckCallback;
        
        this.init();
    }

    /** Constructs a SBC parser object with default values. */
    public SBCParser() {
        this(
                Languages.ENGLISH,
                false,
                new ErrorReceiver() {
                    public void error(String s, SBCParser p) { }
                },
                new URLChecker() {
                    public boolean isValid(String s) { return true; }
                },
                new URLChecker() {
                    public boolean isValid(String s) { return true; }
                }
        );
    }

    /* ACCESSOR SECTION */

    /** @return Returns the language. */
    public int getLanguage() {
        return language;
    }

    /**
     * Sets the language.
     * 
     * It defines the language of your error messages.
     * 
     * @param language the language to set.
     */
    public void setLanguage(int language) {
        this.language = language;
    }

    /** @return Returns the imageSupport. */
    public boolean isImageSupport() {
        return imageSupport;
    }

    /**
     * Sets whether image code is parsed or not.
     * 
     * @param imageSupport the imageSupport to set.
     */
    public void setImageSupport(boolean imageSupport) {
        this.imageSupport = imageSupport;
    }

    /** @return Returns the errorCallback. */
    public ErrorReceiver getErrorCallback() {
        return errorCallback;
    }

    /**
     * Sets the error callback object.
     * 
     * Every time an error occurs while parsing, its error() method is called.
     * 
     * @param errorCallback The errorCallback to set.
     */
    public void setErrorCallback(ErrorReceiver errorCallback) {
        this.errorCallback = errorCallback;
    }

    /** @return Returns the linkcheckCallback. */
    public URLChecker getLinkcheckCallback() {
        return linkcheckCallback;
    }

    /**
     * Sets the linkcheck callback object.
     * 
     * On a given URL its isValid() method has to return true if and only if
     * this URL is valid/good.
     * 
     * @param linkcheckCallback
     *            the linkcheckCallback to set.
     */
    public void setLinkcheckCallback(URLChecker linkcheckCallback) {
        this.linkcheckCallback = linkcheckCallback;
    }

    /** @return Returns the imgcheckCallback. */
    public URLChecker getImgcheckCallback() {
        return imgcheckCallback;
    }

    /**
     * Sets the imgcheck callback object.
     * 
     * On a given URL its isValid() method has to return true if and only if
     * this URL is valid/good.
     * 
     * @param imgcheckCallback
     *            The imgcheckCallback to set.
     */
    public void setImgcheckCallback(URLChecker imgcheckCallback) {
        this.imgcheckCallback = imgcheckCallback;
    }

    /* INTERNAL STUFF */

    /** Prepare for new parser session. */
    private void init() {
        text        = "";
        result      = "";
        attribute   = "";
        errors      = new Vector();
        istack      = new Istack();
        qstack      = 0;
        line        = 0;
    }

    /** Report an error message. */
    private void error(int error) {
        String emsg = Errors.get(error, line, language);
        errors.add(emsg);
        errorCallback(emsg);
    }

    /** Report an error message with an argument string. */
    private void error(int error, String arg) {
        String emsg = Errors.get(error, arg, line, language);
        errors.add(emsg);
        errorCallback(emsg);
    }

    /** Call the error callback. */
    private void errorCallback(String emsg) {
        errorCallback.error(emsg, this);
    }

    /** Call the linkcheck callback. */
    private boolean linkcheckCallback(String url) {
        return linkcheckCallback.isValid(url);
    }

    /** Call the imgcheck callback. */
    private boolean imgcheckCallback(String url) {
        return imgcheckCallback.isValid(url);
    }

    /** Pre parse. */
    private void pre() {
        text = text .replaceAll("&", "&amp;")
                    .replaceAll("\\\\<", "&lt;")
                    .replaceAll("\\\\>", "&gt;")
                    .replaceAll("\"", "&quot;")
                    .replaceAll("[\t ]+", " ")
                    ;
    }

    /** Post parse. */
    private void post() {
        result = result.replaceAll("\\\\([*_<>{}\\[\\]#\\\\])", "$1");
    }

    /** Literal consumer with replacement (ugly, but works) */
    private boolean literal(Pattern token, String replacement) {
        String copy = text;
        Matcher m = token.matcher(text);
        text = m.replaceFirst(replacement);
        boolean success = !text.equals(copy);
        if (! success && token.equals(Tokens.LINEBREAK)) {
            // s/\n/\n/ ?
            Matcher tester = token.matcher(copy);
            copy = tester.replaceFirst("FOO");
            if (! copy.equals(text))
                success = true;
        }
        attribute = (success && m.groupCount() > 0) ? m.group(1) : "";
        return success;
    }

    /** Literal consumer. */
    private boolean literal(Pattern token) {
        return literal(token, "");
    }

    /* PARSER SECTION */

    private String sbc() {
        String block, sbc = "";
        while (!(block = block()).equals("")) {
            sbc += block;
        }
        return sbc;
    }

    private String block() {
        String block = quote();
        if (block.equals("")) block = ulist();
        if (block.equals("")) block = olist();
        if (block.equals("")) block = paragraph();
        return block;
    }

    private String quote() {
        if (!literal(Tokens.QUOTE_START, "\n")) {
            return "";
        }
        line++;

        qstack++;
        String quote = sbc();
        qstack--;

        if (literal(Tokens.QUOTE_END, "\n")) {
            return "<div class=\"quote\">" + "<blockquote>\n" + quote
                    + "</blockquote></div>\n";
        } else if (literal(Tokens.QUOTE_END_CITE)) {
            String cite = inline();
            return "<div class=\"quote\"><cite>" + cite + "</cite>"
                    + "<blockquote>\n" + quote + "</blockquote></div>\n";
        } else {
            error(Errors.NO_QUOTE_END);
            return "<div class=\"quote\">" + "<blockquote>\n" + quote
                    + "</blockquote></div>\n";
        }
    }

    private String ulist() {
        String ulitem, ulist = "";
        while (!(ulitem = ulitem()).equals("")) {
            ulist += ulitem;
        }
        return ulist.equals("") ? "" : ("<ul>\n" + ulist + "</ul>\n");
    }

    private String ulitem() {
        if (!literal(Tokens.UL_BULLET)) {
            return "";
        }
        line++;

        String ulitem = inline();
        return "\t<li>" + ulitem + "</li>\n";
    }

    private String olist() {
        String olitem, olist = "";
        while (!(olitem = olitem()).equals("")) {
            olist += olitem;
        }
        return olist.equals("") ? "" : ("<ol>\n" + olist + "</ol>\n");
    }

    private String olitem() {
        if (!literal(Tokens.OL_BULLET)) {
            return "";
        }
        line++;

        String olitem = inline();
        return "\t<li>" + olitem + "</li>\n";
    }

    private String paragraph() {
        if (!literal(Tokens.LINEBREAK)) {
            return "";
        }
        line++;

        String paragraph = inline();

        if (! (qstack != 0 || literal(Tokens.LINEBREAK, "\n"))) {
            line--;
            return "";
        }
        if (paragraph.matches("\\s*")) {
            return "\n";
        } else {
            return "<p>" + paragraph + "</p>\n";
        }
    }

    private String inline() {
        String emphasis, strong, hyperlink, image, plain, inline = "";
        while (true) {
            if (istack.emphasis == 0
                        && !(emphasis = emphasis()).equals("")) {
                inline += emphasis; continue;
            } else if (istack.strong == 0
                        && !(strong = strong()).equals("")) {
                inline += strong; continue;
            } else if (istack.hyperlink == 0
                        && !(hyperlink = hyperlink()).equals("")) {
                inline += hyperlink; continue;
            } else if (imageSupport
                        && !(image = image()).equals("")) {
                inline += image; continue;
            } else if (!(plain = plain()).equals("")) {
                inline += plain; continue;
            } else {
                break;
            }
        }
        return inline;
    }

    private String emphasis() {
        if (!literal(Tokens.EMPHASIS)) {
            return "";
        }
        istack.emphasis++;

        String emphasis = inline();

        if (!literal(Tokens.EMPHASIS)) {
            error(Errors.NO_EMPHASIS_END);
        }
        istack.emphasis--;
        return emphasis.equals("") ? "" : ("<em>" + emphasis + "</em>");
    }

    private String strong() {
        if (!literal(Tokens.STRONG)) {
            return "";
        }
        istack.strong++;

        String strong = inline();

        if (!literal(Tokens.STRONG)) {
            error(Errors.NO_STRONG_END);
        }
        istack.strong--;
        return strong.equals("") ? "" : ("<strong>" + strong + "</strong>");
    }

    private String hyperlink() {
        if (!literal(Tokens.HYPERLINK_START)) {
            return "";
        }
        istack.hyperlink++;

        String url = attribute;
        String link = inline();
        if (link.matches("\\s*")) {
            link = url;
        }

        if (!literal(Tokens.HYPERLINK_END)) {
            error(Errors.NO_HYPERLINK_END);
        }
        istack.hyperlink--;
        if (linkcheckCallback(url)) {
            return "<a href=\"" + url + "\">" + link + "</a>";
        } else {
            this.error(Errors.FORBIDDEN_URL, url);
            return link;
        }
    }

    private String image() {
        if (!literal(Tokens.IMAGE_START)) {
            return "";
        }

        String url = attribute;
        String alt = "";
        String plain;
        while (!(plain = plain()).equals("")) {
            alt += plain;
        }

        if (!literal(Tokens.IMAGE_END)) {
            error(Errors.NO_IMAGE_END);
        }
        if (imgcheckCallback(url)) {
            return "<img src=\"" + url + "\" alt=\"" + alt + "\">";
        } else {
            error(Errors.FORBIDDEN_URL, url);
            return "";
        }
    }

    private String plain() {
        return literal(Tokens.PLAIN) ? attribute : "";
    }

    /* PUBLIC TRANSLATION SECTION */

    /**
     * SBC translation.
     * 
     * @param text SBC text
     * @return valid HTML block elements representing the given SBC text.
     */
    public String sbc(String text) {
        if (text.matches("\\s*")) {
            return "";
        }
        init();
        this.text = text;
        pre();
        this.text = "\n" + this.text + "\n";
        this.text = this.text.replaceAll("[\r\n]+", "\n");
        result = sbc();
        post();
        result = result.replaceAll("\\\\\n", "<br>");
        if (!this.text.matches("\n*")) {
            error(Errors.UNKNOWN_TOKEN);
        }
        return result;
    }

    /**
     * SBC inline translation.
     * 
     * @param text inline SBC text (only inline elements!)
     * @return valid HTML inline elements representing the given inline SBC text.
     */
    public String sbcInline(String text) {
        if (text.matches("\\s*")) {
            return "";
        }
        init();
        this.text = text;
        pre();
        this.text = this.text.replaceAll("[\r\n]+", " ");
        result = inline();
        post();
        result = result.replaceAll("\\\\\n", "<br>");
        if (!this.text.matches("\n*")) {
            error(Errors.UNKNOWN_TOKEN);
        }
        return result;
    }

    /* PUBLIC ERROR HANDLING SECTION */

    /**
     * Returns whether there are more errors in your SBC (iterator).
     * 
     * @return true if there are more error messages.
     */
    public boolean hasMoreErrors() {
        return !errors.isEmpty();
    }

    /**
     * Get a vector of error messages after translation.
     * 
     * @return a Vector of error messages (String)
     */
    public Vector errors() {
        return (Vector) errors.clone();
    }

    /**
     * Get the next error message (iterator interface with hasMoreErrors).
     * 
     * @return the next error message (String)
     */
    public String nextError() {
        return (String) errors.remove(0);
    }

    /* STATIC METHOD SECTION */

    /**
     * Quoted SBC.
     * 
     * @param sbc your SBC input
     * @return your string quoted
     */
    public static String quote(String sbc) {
        return quote(sbc, "");
    }

    /**
     * Quoted SBC.
     * 
     * @param sbc your SBC input
     * @param cite the author
     * @return your string quoted
     */
    public static String quote(String sbc, String cite) {
        return "[\n" + sbc + "\n]\n";
    }

    /**
     * SBC language description.
     * 
     * @param language your language
     * @return HTML string with SBC translation
     */
    public static String description(int language) {
        SBCResult result = SBCParser.sbcTranslate(Descriptions.get(language));
        for (int i = 0; i < result.errors.size(); i++) {
            System.err.println(result.errors.elementAt(i));
        }
        return result.result;
    }

    /* STATIC TRANSLATION SECTION */

    private static SBCParser initStaticParser() {
        if (!(staticParser instanceof SBCParser)) {
            staticParser = new SBCParser();
            staticParser.setImageSupport(false);
        }
        return staticParser;
    }

    private static int getStaticParserLanguage() {
        return initStaticParser().getLanguage();
    }

    /** Static parser interface: set language to english. */
    public static void english() {
        initStaticParser().setLanguage(Languages.ENGLISH);
    }

    /** Static parser interface: set language to german. */
    public static void german() {
        initStaticParser().setLanguage(Languages.GERMAN);
    }
    
    /**
     * Static parser interface: SBC translation of given text.
     * 
     * @param text your SBC text
     * @return SBCResult with html text as result and an error Vector as errors.
     */
    public static SBCResult sbcTranslate(String text) {
        return new SBCResult(
                    initStaticParser().sbc(text),
                    staticParser.errors()
        );
    }

    /**
     * Static parser interface: SBC inline translation of given text.
     * 
     * @param text your SBC inline text
     * @return SBCResult with inline html as result and an error Vector as errors.
     */
    public static SBCResult sbcTranslateInline(String text) {
        return new SBCResult(
                    initStaticParser().sbcInline(text),
                    staticParser.errors()
        );
    }

    /**
     * Static parser interface: SBC quotes of given text.
     * 
     * @param sbc the sbc text to be quoted
     * @param cite the author
     * @return sbc quoted
     */
    public static String sbcQuote(String sbc, String cite) {
        return quote(sbc, cite);
    }

    /**
     * Static parser interface: SBC quotes of given text.
     * 
     * @param sbc the sbc text to be quoted
     * @return sbc quoted
     */
    public static String sbcQuote(String sbc) {
        return quote(sbc);
    }

    /**
     * Static parser interface: SBC language description
     * 
     * @return HTML string with SBC translation
     */
    public static String sbcDescription() {
        return description(getStaticParserLanguage());
    }
}
